﻿using System.Linq;
using Hims.Api.Models.Account;

namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using AutoMapper;
    using Domain.Helpers;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Models.Patient;
    using Senders;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library;
    using Shared.Library.Enums;
    using Shared.UserModels;
    using Shared.UserModels.Filters;
    using Utilities;

    using Hims.Domain.Configurations;
    using Hims.Api.Helper;
    using Hims.Shared.UserModels.DemandBook;
    using Hims.Api.Models;

    /// <inheritdoc />
    /// <summary>
    /// The demand book controller.
    /// </summary>
    [Authorize]
    [Route("api/demand-book")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class DemandBookController : BaseController
    {
        /// <summary>
        /// Thedemand book services.
        /// </summary>
        private readonly IDemandBookService demandBookService;

        /// <summary>
        /// the patient service
        /// </summary>
        private readonly IPatientService patientServices;

        /// <summary>
        /// The account services.
        /// </summary>
        private readonly IAccountService accountServices;

        /// <summary>
        /// The pharmacy service.
        /// </summary>
        private readonly IPharmacyLogService pharmacyLogService;
        /// <inheritdoc />
        public DemandBookController(IDemandBookService demandBookService, IPatientService patientServices, IAccountService accountServices, IPharmacyLogService pharmacyLogService)
        {
            this.demandBookService = demandBookService;
            this.patientServices = patientServices;
            this.accountServices = accountServices;
            this.pharmacyLogService = pharmacyLogService;
        }

        /// <summary>
        /// The add coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon added successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("insert")]
        [ProducesResponseType(400)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] InsertModel model, [FromHeader] LocationHeader location)
        {
            model = (InsertModel)EmptyFilter.Handler(model);
            model.LocationId = int.Parse(location.LocationId);
            var response = await this.demandBookService.AddAsync(model);
            if (response > 0)
            {           
                var pharmacyLogModel = new PharmacyLogModel
                {
                   
                    AccountId = model.CreatedBy,
                    PharmacyLogTypeId = (int)PharmacyLogTypes.Demand_Book,
                    LogFrom = (short)model.LoginRoleId,
                    LogDate = DateTime.Now,                
                   // LogDescription = $@"Demand Book Request added Successfully. Product Name :'{model.ProductName}'<br/>Required Quantity:'{model.RequiredQty}'"
                    LogDescription = $@"Demand Book Request added Successfully. Product Name :<b>'{model.ProductName}'</b><br/> with  quantity <b>'{model.RequiredQty}'</b> has added into Demand book for Patient <b>'{model.PatientName}'</b> with Mobile number <b>'{model.Mobile}'</b>"
                };
                await this.pharmacyLogService.LogAsync(pharmacyLogModel);
            }
            return this.Success(response);
        }

        /// <summary>
        /// The add coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon added successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(400)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] ViewModel model, [FromHeader] LocationHeader location)
        {
            model = (ViewModel)EmptyFilter.Handler(model);
            model.LocationId = int.Parse(location.LocationId);
            var response = await this.demandBookService.FetchAsync(model);
            return response == null ? this.ServerError() : this.Success(response);
        }

        /// <summary>
        /// The add pharmacy product.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("update")]
        public async Task<ActionResult> UpdateDemandRecord([FromBody] ViewModel model)
        {
            model = (ViewModel)EmptyFilter.Handler(model);
            var response = await this.demandBookService.UpdateDemandRecord(model);
            var pharmacyLogModel = new PharmacyLogModel
            {
                AccountId = model.LoginAccountId,
                PharmacyLogTypeId = (int)PharmacyLogTypes.Demand_Book,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.Now,
                LogDescription = $@"Demand Book Request updated Successfully Product Name :'{model.ProductName}'<br/>Required Quantity:'{model.RequiredQty}'"
            };
            await this.pharmacyLogService.LogAsync(pharmacyLogModel);
            return this.Success(response);
        }
    }
}